let mainVideoContainer;
window.__videoBound = false;

document.addEventListener('DOMContentLoaded', async () => {
  try {
    mainVideoContainer = document.querySelector('.main-video-container');
    if (!mainVideoContainer) throw new Error('Main video container not found');
    await preloadVideos();
  } catch (error) {
    console.error('Video initialization error:', error);
  }
});

// ---------- PRELOAD ----------
async function preloadVideos() {
  const videoSources = ['videos/001_V.mp4', 'videos/PAUSE.mp4'];
  const loading = document.querySelector('.loader');

  try {
    await Promise.all(
      videoSources.map(src => new Promise((resolve, reject) => {
        const v = document.createElement('video');
        v.preload = 'auto';
        v.onloadeddata = () => resolve();
        v.onerror = () => reject(`Failed to load ${src}`);
        v.src = src;
      }))
    );
  } catch (e) {
    console.error('Video preload failed:', e);
  } finally {
    document.body.classList.add('loaded');
    setTimeout(() => loading && loading.remove(), 500);
    initializeVideos();
  }
}

// ---------- INITIALIZE ----------
function initializeVideos() {
  const pauseVideo  = document.getElementById('video-pause');
  const mainVideo   = document.getElementById('video-file');
  const videoMain   = document.querySelector('.video-main');
  const playToggle  = document.getElementById('play-toggle');

  if (!pauseVideo || !mainVideo || !videoMain || !playToggle) return;
  if (window.__videoBound) return; window.__videoBound = true;

  showPauseLayer();

  const setBtnState = (isPlaying) => {
    playToggle.setAttribute('aria-label', isPlaying ? 'Pause' : 'Play');
    playToggle.classList.toggle('is-playing', isPlaying);
  };

  playToggle.addEventListener('click', () => {
    if (mainVideo.ended) mainVideo.currentTime = 0;

    if (mainVideo.paused) {
      showMainLayer();
      mainVideo.play().catch(() => {});
      setBtnState(true);
    } else {
      mainVideo.pause();
      showPauseLayer();
      setBtnState(false);
    }
  });

  mainVideo.addEventListener('play', () => {
    showMainLayer();
    setBtnState(true);
  });

  mainVideo.addEventListener('pause', () => {
    if (!mainVideo.ended) {
      showPauseLayer();
      setBtnState(false);
    }
  });

  mainVideo.addEventListener('ended', () => {
    showPauseLayer();
    setBtnState(false);
  });

  function showMainLayer() {
    pauseVideo.pause();
    pauseVideo.currentTime = 0;
    pauseVideo.style.opacity = '0';
    gsap.to('.video-main', { opacity: 1, duration: 0.25, ease: 'power1.out', pointerEvents: 'auto' });
  }

  function showPauseLayer() {
    gsap.to('.video-main', { opacity: 0, duration: 0.25, ease: 'power1.out', pointerEvents: 'none' });
    pauseVideo.style.opacity = '1';
    pauseVideo.play().catch(()=>{});
  }
}
