let mainVideoContainer;
window.__videoBound = false;

document.addEventListener('DOMContentLoaded', async () => {
  try {
    mainVideoContainer = document.querySelector('.main-video-container');
    if (!mainVideoContainer) throw new Error('Main video container not found');
    await preloadVideos();
  } catch (error) {
    console.error('Video initialization error:', error);
  }
});

// ---------- PRELOAD ----------
async function preloadVideos() {
  const videoSources = ['videos/001_V.mp4', 'videos/PAUSE.mp4', 'videos/ROCKET.mp4'];
  const loading = document.querySelector('.loader');

  try {
    await Promise.all(
      videoSources.map(src => new Promise((resolve, reject) => {
        const v = document.createElement('video');
        v.preload = 'auto';
        v.onloadeddata = () => resolve();
        v.onerror = () => reject(`Failed to load ${src}`);
        v.src = src;
      }))
    );
  } catch (e) {
    console.error('Video preload failed:', e);
  } finally {
    document.body.classList.add('loaded');
    setTimeout(() => loading && loading.remove(), 500);
    initializeVideos();
    initializeStartupModal(); // инициализация модалки
  }
}

// ---------- INITIALIZE MAIN VIDEOS ----------
function initializeVideos() {
  const pauseVideo  = document.getElementById('video-pause');
  const mainVideo   = document.getElementById('video-file');
  const videoMain   = document.querySelector('.video-main');
  const playToggle  = document.getElementById('play-toggle');

  if (!pauseVideo || !mainVideo || !videoMain || !playToggle) return;
  if (window.__videoBound) return; window.__videoBound = true;

  showPauseLayer();

  const setBtnState = (isPlaying) => {
    playToggle.setAttribute('aria-label', isPlaying ? 'Pause' : 'Play');
    playToggle.classList.toggle('is-playing', isPlaying);
  };

  playToggle.addEventListener('click', () => {
    if (mainVideo.ended) mainVideo.currentTime = 0;

    if (mainVideo.paused) {
      showMainLayer();
      mainVideo.play().catch(() => {});
      setBtnState(true);
    } else {
      mainVideo.pause();
      showPauseLayer();
      setBtnState(false);
    }
  });

  mainVideo.addEventListener('play', () => {
    showMainLayer();
    setBtnState(true);
  });

  mainVideo.addEventListener('pause', () => {
    if (!mainVideo.ended) {
      showPauseLayer();
      setBtnState(false);
    }
  });

  mainVideo.addEventListener('ended', () => {
    showPauseLayer();
    setBtnState(false);
  });

  function showMainLayer() {
    pauseVideo.pause();
    pauseVideo.currentTime = 0;
    pauseVideo.style.opacity = '0';
    gsap.to('.video-main', { opacity: 1, duration: 0.25, ease: 'power1.out', pointerEvents: 'auto' });
  }

  function showPauseLayer() {
    gsap.to('.video-main', { opacity: 0, duration: 0.25, ease: 'power1.out', pointerEvents: 'none' });
    pauseVideo.style.opacity = '1';
    pauseVideo.play().catch(()=>{});
  }
}

// ---------- STARTUP MODAL ----------
function initializeStartupModal() {
  const openBtn   = document.getElementById('startup-open');
  const modal     = document.getElementById('startup-modal');
  const backdrop  = modal?.querySelector('.startup-modal__backdrop');
  const closeBtn  = document.getElementById('startup-close');
  const videoEl   = document.getElementById('startup-video');
  const caption   = document.getElementById('startup-caption');
  const cta       = document.getElementById('startup-cta');
  const player    = document.getElementById('startup-player');

  // Новые элементы для управления звуком:
  const soundBtn  = document.getElementById('startup-sound-toggle');
  const soundIcon = soundBtn?.querySelector('i');

  if (!openBtn || !modal || !backdrop || !closeBtn || !videoEl || !caption || !cta || !player || !soundBtn) return;

  const mainVideo = document.getElementById('video-file');

  function lockScroll(lock) {
    document.body.classList.toggle('startup-modal-open', lock);
  }

  function resetState() {
    gsap.set([cta], { autoAlpha: 0, display: 'none' });
    gsap.set([player, caption], { autoAlpha: 1, display: 'block' });
    // Всегда возвращаем mute
    videoEl.pause();
    videoEl.currentTime = 0;
    videoEl.muted = true;
    videoEl.volume = 0;
    // Сброс иконки/состояния кнопки
    soundBtn.setAttribute('aria-pressed', 'false');
    if (soundIcon) soundIcon.className = 'bi bi-volume-mute';
    modal.querySelectorAll('.startup-cta__item').forEach(a => a.setAttribute('tabindex', '-1'));
  }

  // --- открытие модалки (не включаем звук) ---
  function openModal(e) {
    if (e) e.preventDefault();
    resetState();
    modal.classList.add('is-open');
    lockScroll(true);
    closeBtn.focus({ preventScroll: true });

    try { mainVideo && mainVideo.pause(); } catch {}

    // Запускаем БЕЗ звука (жест пользователя — сам клик по open)
    videoEl.currentTime = 0;
    videoEl.play().catch(err => console.warn('Autoplay (muted) blocked:', err));
  }

  function closeModal() {
    modal.classList.remove('is-open');
    lockScroll(false);
    resetState();
  }

  openBtn.addEventListener('click', openModal);
  backdrop.addEventListener('click', (e)=>{ if (e.target.dataset.close === 'true') closeModal(); });
  closeBtn.addEventListener('click', closeModal);

  document.addEventListener('keydown', (e) => {
    if (!modal.classList.contains('is-open')) return;
    if (e.key === 'Escape') closeModal();
  });

  videoEl.addEventListener('ended', () => {
    const tl = gsap.timeline();
    tl.to(caption, { autoAlpha: 0, duration: 0.25, ease: 'power1.out' }, 0)
      .to(player,  { autoAlpha: 0, duration: 0.35, ease: 'power1.out' }, 0)
      .set([caption, player], { display: 'none' })
      .set(cta, { display: 'grid' })
      .to(cta, { autoAlpha: 1, duration: 0.35, ease: 'power1.out' });

    modal.querySelectorAll('.startup-cta__item').forEach(a => a.setAttribute('tabindex', '0'));
  });

  // ---- Управление звуком только через кнопку ----
  soundBtn.addEventListener('click', () => {
    const willUnmute = videoEl.muted; // если сейчас mute, то включаем звук
    videoEl.muted  = !willUnmute;
    videoEl.volume = willUnmute ? 1 : 0;
    soundBtn.setAttribute('aria-pressed', String(willUnmute));
    if (soundIcon) {
      soundIcon.className = willUnmute ? 'bi bi-volume-up' : 'bi bi-volume-mute';
    }
    // если видео на паузе, пробуем продолжить воспроизведение
    if (videoEl.paused) videoEl.play().catch(()=>{});
  });

  // Клик по самому видео НЕ меняет звук
  videoEl.addEventListener('click', (e) => e.preventDefault());
}

